# -*- coding: UTF-8 -*-

import os
import yaml
import unittest
import pytest
from collections.abc import Iterable, Mapping

EXCLUDED_DIRS = ['./playbooks/templates/prometheus', './output']
ADDITIONAL_PATHS = ['./group_vars', './host_vars']

def any_constructor(loader, tag_suffix, node):
    if isinstance(node, yaml.MappingNode):
        return loader.construct_mapping(node)
    if isinstance(node, yaml.SequenceNode):
        return loader.construct_sequence(node)
    return loader.construct_scalar(node)


def is_excluded(path, file):
    for excluded_dir in EXCLUDED_DIRS:
        if path == excluded_dir or path.startswith(excluded_dir + '/'):
            return True
    return False


def has_invalid_chars(s: str) -> bool:
    if not isinstance(s, str):
        return False
    for ch in s:
        codepoint = ord(ch)
        if codepoint <= 0x20:
            return True
        if codepoint in (0x85, 0xa0):
            return True
        # please don't use unicode codepoints above 0x2000 for now :)
        if codepoint >= 0x2000:
            return True
    return False


def check_keys(element, file):
    if isinstance(element, Mapping):
        for k, v in element.items():
            if isinstance(k, str):
                assert not has_invalid_chars(k), f'key "{k}" in {file} had an invalid character'
            check_keys(v, file)
    elif isinstance(element, str):
        pass
    elif isinstance(element, Iterable):
        for child in element:
            check_keys(child, file)


def all_yamls():
    for dirpath, _, filenames in os.walk('.'):
        for file in filenames:
            if is_excluded(dirpath, file):
                continue
            if file.endswith('.yml') or file.endswith('.yaml') or dirpath in ADDITIONAL_PATHS:
                path = os.path.join(dirpath, file)
                yield path

yaml.add_multi_constructor('', any_constructor, Loader=yaml.SafeLoader)

@pytest.mark.parametrize("path", all_yamls())
def testYamlValid(path):
    with open(path, 'r', encoding='utf8') as stream:
        result = yaml.load(stream, Loader=yaml.SafeLoader)
        check_keys(result, path)
