# -*- coding: UTF-8 -*-

from ansibleext.facade import AnsibleLibraryFacade, InventoryFacadeBase
from .facade import HostFacade
import unittest
import glob
import re

from typing import Optional, List, Any, Set, Dict, Iterable

class InventoryCache:

    def __init__(self):
        self._inventories = {}

    def load_inventory(self, ansible: AnsibleLibraryFacade, inventory_file: str) -> InventoryFacadeBase:
        if inventory_file not in self._inventories:
            self._inventories[inventory_file] = ansible.load_inventory(inventory_file)
        return self._inventories[inventory_file]

INVENTORY_CACHE = InventoryCache()

class InventoryBase(unittest.TestCase):
    def verifyPropertyExists(self, inventory: InventoryFacadeBase, host: HostFacade, key: str):
        variables = inventory.get_variables(host.get_name())
        self.assertIn(key, variables, msg='{0} not found in {1} at {2}'.format(key, host, inventory))
        return variables[key]

    def assertDictContainsItems(self, variableDict: Any, subsetDict: dict, msg: Optional[str] = None):
        self.assertIsInstance(variableDict, dict)
        self.assertIsInstance(subsetDict, dict)
        if not (subsetDict.items() <= variableDict.items()):
            self.fail(self._formatMessage(msg, f'Expected {variableDict} to contain all elements of {subsetDict}, but some were missing or their values not equal.'))

    def isHostInAnyGroups(self, inventory: InventoryFacadeBase, host: HostFacade, groupNames: Iterable[str]) -> bool:
        groups = self.getGroups(inventory, host)
        return any(groupName in groups for groupName in groupNames)

    def getGroups(self, inventory: InventoryFacadeBase, host: HostFacade) -> List[str]:
        groups = self.getVariable(inventory, host, 'group_names')
        assert isinstance(groups, list)
        assert all(isinstance(x, str) for x in groups)
        return groups

    def groupNamesContain(self, inventory: InventoryFacadeBase, host: HostFacade, key: str) -> bool:
        groups = self.getGroups(inventory, host)
        return any(key in groupName for groupName in groups)

    def getGroupsWithNameMatching(self, inventory: InventoryFacadeBase, host: HostFacade, regex: str, ignoredGroups: Iterable[str]) -> Set[str]:
        groups = self.getGroups(inventory, host)
        matchingHosts = set()
        for group in groups:
            if re.search(regex, group):
                if group not in ignoredGroups:
                    matchingHosts.add(group)
        return matchingHosts

    def isHostManuallySetup(self, inventory: InventoryFacadeBase, host: HostFacade) -> bool:
        manual_setup = self.getVariable(inventory, host, "manual_setup")
        if (manual_setup is not None) and (manual_setup.lower() in ('true', 'yes')):
            return True
        return False

    def getVariable(self, inventory: InventoryFacadeBase, host: HostFacade, key: str) -> Optional[Any]:
        return self.getVariableForName(inventory, host.get_name(), key)

    def getVariableForName(self, inventory: InventoryFacadeBase, hostName: str, key: str) -> Optional[Any]:
        self.assertIsInstance(hostName, str)
        variables = inventory.get_variables(hostName)
        if key not in variables:
            return None
        return variables[key]

    def getVariablesForHostName(self, inventory: InventoryFacadeBase, hostName: str) -> Dict[str, Any]:
        self.assertIsInstance(hostName, str)
        return inventory.get_variables(hostName)

    def inventories_hosts(self):
        inventory_files = glob.glob('inventory-*')
        for inventory_file in inventory_files:
            inventory = INVENTORY_CACHE.load_inventory(self.ansible, inventory_file)
            for host in inventory.get_hosts('*'):
                name = host.get_name()
                if name == 'localhost':
                    continue
                yield (inventory, host)

    def inventories_hosts_onlystable(self):
        inventory_files = glob.glob('inventory-stable')
        for inventory_file in inventory_files:
            inventory = INVENTORY_CACHE.load_inventory(self.ansible, inventory_file)
            for host in inventory.get_hosts('*'):
                name = host.get_name()
                if name == 'localhost':
                    continue
                yield (inventory, host)

    def inventory(self, inventory_file) -> InventoryFacadeBase:
        return INVENTORY_CACHE.load_inventory(self.ansible, inventory_file)

    def inventories(self) -> List[InventoryFacadeBase]:
        inventory_files = glob.glob('inventory-*')
        return [INVENTORY_CACHE.load_inventory(self.ansible, i) for i in inventory_files if i != 'inventory-balu']

    def inventoriesHostsInAnyGroups(self, groupNames: List[str]):
        #self.assertTrue(False, 'groups: %s' % groupNames)
        for inventory, host in self.inventories_hosts_onlystable():
            #self.assertTrue(False, 'host: %s' % host)
            hostInGroups = self.isHostInAnyGroups(inventory, host, groupNames)
            if hostInGroups:
                yield (inventory, host)

    def inventories_groups_onlystable(self) -> List[str]:
        inventory_files = glob.glob('inventory-stable')
        if len(inventory_files) == 0:
            return []
        for inventory_file in inventory_files:
            inventory = INVENTORY_CACHE.load_inventory(self.ansible, inventory_file)
        inventory_groups: Set[str] = set()
        for host in inventory.get_hosts('*'):
            groups = self.getGroups(inventory, host)
            inventory_groups.update(groups)
        return list(inventory_groups)

    def setUp(self):
        self.ansible = AnsibleLibraryFacade('.')
        self.longMessage = True
        self.maxDiff = None
