# -*- coding: UTF-8 -*-

import sys
import os
import uuid
import yaml
import ansibleext.yamlsetup as yamlsetup

def log(text):
    print(text)
    sys.stdout.flush()

class AsbruOutputter:
    def __init__(self, inventory):
        self.inventory = inventory
        self.hosts = dict()
        self.groups = dict()
        self.hostsByUUID = dict()
        self.ROOT_GROUP_NAME = '__PAC__EXPORTED__'
        self.rootChildren = dict()

    def output(self, objectname, pacYml):
        with open(pacYml, "w") as f:
            self.writeOutput(objectname, f)

    def generateUuid(self):
        return uuid.uuid4()

    def getOrError(self, host, dictionary, keys):
        ret = self.getOrDefault(dictionary, keys, None)
        if ret is None:
            raise ValueError('Host ' + host.get_name() + ' has no value for any of ' + str(keys))
        return ret

    def getOrDefault(self, dictionary, keys, default):
        for key in keys:
            if key in dictionary:
                return dictionary[key]
        return default

    def generateName(self, host, portForwardEnabled, userName):
        if portForwardEnabled:
            return host.get_name() + ' portfwd'
        else:
            return host.get_name() + ' (' + userName + ')'

    def generateTitle(self, host, host_variables, portForwardEnabled, userName):
        baseName = self.generateName(host, portForwardEnabled, userName)
        if 'parent' in host_variables:
            return host_variables['parent'] + ' - ' + baseName
        return baseName

    def isStable(self, groups):
        if len(groups) == 0:
            return False
        for group in groups:
            if group.get_name().startswith('stable'):
                return True
            if self.isStable(group.get_parent_groups()):
                return True
        return False

    def generateColor(self, host, host_variables):
        if self.isStable(host.get_groups()):
            return '#550000000000'
        else:
            return '#000000000000'

    def addLocalPortForward(self, options, port):
        options.extend(['-L', 'localhost/' + str(port) + '/localhost/' + str(port)])

    def getOrWarning(self, host, mapVar, keySequence):
        current = mapVar
        current_key = []
        for key in keySequence:
            if not isinstance(current, dict):
                log('Warning: key ' + '.'.join(current_key) + ' required to access path ' + '.'.join(keySequence) + ' not a map for host ' + host.get_name())
                raise ValueError
            current_key.append(key)
            if not key in current:
                log('Warning: key ' + '.'.join(current_key) + ' required to access path ' + '.'.join(keySequence) + ' not found for host ' + host.get_name())
                raise ValueError
            current = current[key]
        return current

    def addItrackPortForward(self, options, host, host_variables):
        try:
            rmi_port = self.getOrWarning(host, host_variables, ['itrack_config', 'jmx', 'rmi_port'])
            rmiregistry_port = self.getOrWarning(host, host_variables, ['itrack_config', 'jmx', 'rmiregistry_port'])
            self.addLocalPortForward(options, rmi_port)
            self.addLocalPortForward(options, rmiregistry_port)
        except ValueError:
            pass
        self.addLocalPortForward(options, 8000)

    def isAnyPortForwarded(self, host, host_variables):
        options = self.getSshOptions(host, host_variables, True)
        return '-L' in options

    def getSshOptions(self, host, host_variables, portForwardEnabled):
        options = ['-2', '-x', '-A']
        if portForwardEnabled:
            if 'itrack_config' in host_variables:
                self.addItrackPortForward(options, host, host_variables)
        return options

    def generateSshOptions(self, host, host_variables, portForwardEnabled):
        options = self.getSshOptions(host, host_variables, portForwardEnabled)
        return ' ' + ' '.join(options)

    def getConfiguredUsers(self, host, host_variables):
        users = ['root']
        if 'authpersonmap' in host_variables:
            rawlist = host_variables['authpersonmap']
            if rawlist is not None:
                personmap = rawlist.values()
                splitted_personmap = []
                for x in personmap:
                    splitted_personmap = splitted_personmap + x.split(",")
                    if len(personmap) != 0:
                        tmp = users + splitted_personmap
                        users = list(dict.fromkeys(tmp))
        return users

    def createHost(self, host, host_variables, portForwardEnabled, userName):
        title = self.generateTitle(host, host_variables, portForwardEnabled, userName)
        ret = dict()
        ret['_is_group'] = 0
        ret['_protected'] = 0
        ret['KPX title regexp'] = '.*' + title + '.*'
        ret['auth fallback'] = 1
        ret['auth type'] = 'publickey'
        ret['autoreconnect'] = ''
        ret['autossh'] = ''
        ret['children'] = dict()
        ret['cluster'] = []
        ret['description'] = 'Connection with ' + host.get_name() + ' as user ' + userName
        ret['embed'] = 0
        ret['expect'] = []
        ret['favourite'] = 0
        ret['infer from KPX where'] = 3
        ret['infer user pass from KPX'] = ''
        ret['ip'] = self.getOrDefault(host_variables, ['ansible_host'], host.get_name())
        ret['local after'] = []
        ret['local before'] = []
        ret['local connected'] = []
        ret['mac'] = ''
        ret['macros'] = []
        ret['method'] = 'SSH'
        ret['name'] = self.generateName(host, portForwardEnabled, userName)
        ret['options'] = self.generateSshOptions(host, host_variables, portForwardEnabled)
        ret['original_parent'] = '__PAC__ROOT__'
        ret['parent'] = self.ROOT_GROUP_NAME
        ret['pass'] = ''
        ret['passphrase'] = ''
        ret['passphrase user'] = userName
        ret['port'] = self.getOrDefault(host_variables, ['ansible_port'], 22)
        ret['prepend command'] = ''
        ret['proxy ip'] = ''
        ret['proxy pass'] = ''
        ret['proxy port'] = 8080
        ret['proxy user'] = ''
        ret['public key'] = os.path.expanduser('~') + '/.ssh/id_rsa'
        ret['quote command'] = ''
        ret['remove control chars'] = ''
        ret['save session logs'] = ''
#        ret['screenshots'] = '~'
        ret['search pass on KPX'] = 0
        ret['send slow'] = 0
        ret['send string active'] = ''
        ret['send string every'] = 60
        ret['send string intro'] = 1
        ret['send string txt'] = ''
        ret['session log pattern'] = '<UUID>_<NAME>_<DATE_Y><DATE_M><DATE_D>_<TIME_H><TIME_M><TIME_S>.txt'
        ret['session logs amount'] = 10
        ret['session logs folder'] = os.path.expanduser('~') + '/.config/pac/session_logs'
        ret['startup launch'] = ''
        ret['startup script'] = ''
        ret['startup script name'] = ''
        terminal_options = dict()
        ret['terminal options'] = terminal_options
        terminal_options['audible bell'] = ''
        terminal_options['back color'] = self.generateColor(host, host_variables)
        terminal_options['bold color'] = '#cc62cc62cc62'
        terminal_options['bold color like text'] = 1
        terminal_options['command prompt'] = r'[#%\$>]|\:\/\s*$'
        terminal_options['cursor shape'] = 'block'
        terminal_options['disable ALT key bindings'] = ''
        terminal_options['disable CTRL key bindings'] = ''
        terminal_options['disable SHIFT key bindings'] = ''
        terminal_options['open in tab'] = 1
        terminal_options['password prompt'] = "([p|P]ass|[p|P]ass[w|W]or[d|t]|Enter passphrase for key '.+'):\\s*$"
        terminal_options['tab back color'] = '#000000000000'
        terminal_options['terminal backspace'] = 'auto'
        terminal_options['terminal character encoding'] = 'UTF-8'
        terminal_options['terminal emulation'] = 'xterm'
        terminal_options['terminal font'] = 'Monospace 9'
        terminal_options['terminal scrollback lines'] = 5000
        terminal_options['terminal select words'] = '-.:_/'
        terminal_options['terminal transparency'] = 0
        terminal_options['terminal window hsize'] = 800
        terminal_options['terminal window vsize'] = 600
        terminal_options['text color'] = '#cc62cc62cc62'
        terminal_options['timeout command'] = 40
        terminal_options['timeout connect'] = 40
        terminal_options['use personal settings'] = 1
        terminal_options['use tab back color'] = ''
        terminal_options['username prompt'] = r'([l|L]ogin|[u|u]suario|[u|U]ser-?[n|N]ame|[u|U]ser):\s*$'
        terminal_options['visible bell'] = ''
        ret['title'] = title
        ret['use prepend command'] = ''
        ret['use proxy'] = 0
        ret ['use sudo'] = 0
        ret['user'] = ''
        ret['variables'] = []
        uuid = str(self.generateUuid())
        self.rootChildren[uuid] = 1
        self.hosts[uuid] = ret
        return uuid

    def createRoot(self):
        ret = dict()
        ret['children'] = self.rootChildren
        self.groups[self.ROOT_GROUP_NAME] = ret

    def writeOutput(self, objectname, f):
        hosts = self.inventory.get_hosts(objectname)
        if hosts is None or len(hosts) == 0:
            raise ValueError("No host matched: " + objectname)
        self.createRoot()
        for host in hosts:
            log("processing host " + host.get_name())
            host_variables = self.inventory.get_variables(host.get_name())
            users = self.getConfiguredUsers(host, host_variables)
            forwarding = self.isAnyPortForwarded(host, host_variables)
            for user in users:
                hostUuid = self.createHost(host, host_variables, False, user)
            if forwarding:
                hostUuid = self.createHost(host, host_variables, True, 'root')

        f.write("---\n")
        yamlsetup.add_ansible_representers()
        f.write(yaml.safe_dump(self.hosts, default_flow_style=False, default_style=None, line_break=False, width=1000000))
        f.write(yaml.safe_dump(self.groups, default_flow_style=False, default_style=None, line_break=False, width=1000000))
