# -*- coding: UTF-8 -*-

from collections import OrderedDict
from ansibleext import yamlout
import ipaddress
import re

class VarDump:
    def __init__(self, filter_vars=[], verbose=0, groups_only=False):
        self.filter_vars = filter_vars
        self.verbose = verbose
        self.groups_only = groups_only

    def print_trees(self, inventories, objectname):
        for inventory in inventories:
            self.print_tree(inventory, objectname)

    def print_tree(self, inventory, objectname):
        host = inventory.get_host_or_none(objectname)
        if host is not None:
            self.print_tree_of_host(inventory, host)
            return

        group = inventory.get_group_or_none(objectname)
        if group is not None:
            self.print_tree_of_group(inventory, group)
            return

        hosts = inventory.get_hosts(objectname)
        for host in sorted(hosts, key=lambda host: host.get_name()):
            self.print_tree_of_host(inventory, host)

    def delete_vars(self, data):
        auto_vars = ['inventory_hostname', 'group_names', 'inventory_hostname_short', 'groups',
                     'playbook_dir', 'omit', 'inventory_dir', 'inventory_file', 'ansible_version', 'ansible_playbook_python',
                     'ansible_facts']
        to_be_deleted = auto_vars
        if self.filter_vars is not None:
            for var in data:
                if var not in self.filter_vars:
                    to_be_deleted.append(var)

        for var in to_be_deleted:
            if var in data:
                del data[var]

    def print_tree_of_group(self, inventory, group):
        probe_host = inventory.create_host("probe_host_" + group.get_name(), group)
        group_variables = inventory.get_variables(probe_host.get_name())
        self.delete_vars(group_variables)
        data = OrderedDict({
            'group': group.get_name()
        })
        if self.verbose > 0:
            data['depth'] = group.get_depth()
        if not self.groups_only:
            if self.verbose > 1:
                data['hosts'] = sorted([childhost.get_name() for childhost in group.get_hosts()])
                data['parent_groups'] = sorted([parentgroup.get_name() for parentgroup in group.get_parent_groups()])
                data['child_groups'] = sorted([childgroup.get_name() for childgroup in group.get_child_groups()])
            data['vars'] = group_variables
        yamlout.prettydump(data)

    def print_trees_of_hosts_groups(self, inventory, host):
        for group in sorted(inventory.groups_for_host(host.get_name()), key=lambda g: g.get_depth(), reverse=True):
            print('---------------')
            self.print_tree_of_group(inventory, group)

    def print_tree_of_host(self, inventory, host):
        host_variables = inventory.get_variables(host.get_name())
        self.delete_vars(host_variables)
        if not self.groups_only:
            data = OrderedDict({
                'host': host.get_name()
            })
            if self.verbose > 1:
                data['groups'] = [parentgroup.get_name() for parentgroup in host.get_groups()]
            data['vars'] = host_variables
            yamlout.prettydump(data)
        if self.verbose > 0 or self.groups_only:
            self.print_trees_of_hosts_groups(inventory, host)

def first_existing(variables, list_of_keys, default):
    for key in list_of_keys:
        value = variables.get(key)
        if value:
            return value
    return default

def ip_status(inventory, hostpattern):
    used_ips = {}
    used_networks = set()
    ipv4_regex = re.compile(r"[ ,;]?([0-9]+\.[0-9]+\.[0-9]+\.[0-9]+)[ ,;]?")
    for host in inventory.get_hosts(hostpattern):
        variables = inventory.get_variables(host.get_name())
        ip = variables.get('primary_ip')
        netmask = variables.get('primary_netmask')
        for k, v in variables.items():
            if isinstance(v, str):
                match = ipv4_regex.match(v)
                if match:
                    ipv4 = ipaddress.IPv4Address(match.group(1))
                    if ipv4 not in used_ips:
                        used_ips[ipv4] = 'variable ' + k
        if ip:
            ipv4 = ipaddress.IPv4Address(ip)
            used_ips[ipv4] = first_existing(variables, ['hostname', 'inventory_hostname', 'ansible_host'], '?')
            if netmask:
                network = ipaddress.IPv4Network((ip, netmask), strict=False)
                used_networks.add(network)

    whole_groups = ('*', 'all', 'sibrik_net', 'ilka_net', 'extended_devnet')
    everyHost = hostpattern in whole_groups
    for network in used_networks:
        for ip in network:
            if ip in used_ips:
                print('%s used by %s' % (ip, used_ips[ip]))
            else:
                if everyHost:
                    print('%s unused' % (ip))
    if not everyHost:
        print('No reliable information about unused IPs, use any of {} for host filter'.format(whole_groups))
