# -*- coding: UTF-8 -*-

import sys
import os
import distro # type: ignore

def log(text):
    print(text)
    sys.stdout.flush()

class SSHConfOutputter:
    def __init__(self, inventory, coloring):
        self.inventory = inventory
        self.coloring = coloring

    def output(self, objectname, configFile):
        with open(configFile, "w") as f:
            self.writeOutput(objectname, f)

    def getOrDefault(self, dictionary, keys, default):
        for key in keys:
            if key in dictionary:
                return dictionary[key]
        return default

    def generateName(self, host, portForwardEnabled, userName):
        if portForwardEnabled:
            return host.get_name() + ':portfwd:' + userName
        else:
            return host.get_name() + ':' + userName

    def isStable(self, groups):
        if len(groups) == 0:
            return False
        for group in groups:
            if group.get_name().startswith('stable'):
                return True
            if self.isStable(group.get_parent_groups()):
                return True
        return False

    def generateColor(self, host, host_variables):
        if self.coloring and self.isStable(host.get_groups()):
            return ['LocalCommand xdotool key Alt+Page_Down']
        else:
            return []

    def portforwardOptions(self, host, host_variables):
        portforwardOptions = []
        if 'itrack_config' in host_variables:
            self.addItrackPortForward(portforwardOptions, host, host_variables)
        if 'ssh_portforward' in host_variables:
            for item in host_variables['ssh_portforward']:
                forward = host_variables['ssh_portforward'][item]
                self.addLocalPortForward(portforwardOptions, forward['localport'], forward['remoteport'])

        return portforwardOptions

    def addLocalPortForward(self, options, localport, remoteport):
        options.append('LocalForward ' + str(localport) + ' localhost:' + str(remoteport))

    def getOrWarning(self, host, mapVar, keySequence):
        current = mapVar
        current_key = []
        for key in keySequence:
            if not isinstance(current, dict):
                log('Warning: key ' + '.'.join(current_key) + ' required to access path ' + '.'.join(keySequence) + ' not a map for host ' + host.get_name())
                raise ValueError
            current_key.append(key)
            if not key in current:
                log('Warning: key ' + '.'.join(current_key) + ' required to access path ' + '.'.join(keySequence) + ' not found for host ' + host.get_name())
                raise ValueError
            current = current[key]
        return current

    def addItrackPortForward(self, options, host, host_variables):
        try:
            rmi_port = self.getOrWarning(host, host_variables, ['itrack_config', 'jmx', 'rmi_port'])
            rmiregistry_port = self.getOrWarning(host, host_variables, ['itrack_config', 'jmx', 'rmiregistry_port'])
            self.addLocalPortForward(options, rmi_port, rmi_port)
            self.addLocalPortForward(options, rmiregistry_port, rmiregistry_port)
        except ValueError:
            pass
        self.addLocalPortForward(options, 8000, 8000)

    def isAnyPortForwarded(self, host, host_variables):
        if self.portforwardOptions(host, host_variables):
            return True
        else:
            return False

    def getConfiguredUsers(self, host, host_variables):
        users = ['root']
        if 'authpersonmap' in host_variables:
            rawlist = host_variables['authpersonmap']
            if rawlist is not None:
                personmap = rawlist.values()
                splitted_personmap = []
                for x in personmap:
                    splitted_personmap = splitted_personmap + x.split(",")
                    if len(personmap) != 0:
                        tmp = users + splitted_personmap
                        users = list(dict.fromkeys(tmp))
        return users 

    def createHost(self, f, host, host_variables, portForwardEnabled, userName):
        options = ['',
                'Host ' + self.generateName(host, portForwardEnabled, userName),
                'HostName ' + self.getOrDefault(host_variables, ['ansible_host'], host.get_name()),
                'User ' + userName,
        ]
        options.extend(self.generateColor(host, host_variables))
        if portForwardEnabled:
            options.extend(self.portforwardOptions(host, host_variables))
        options.append('')
        f.write(os.linesep.join(options))

    def createHosts(self, f, host):
        host_variables = self.inventory.get_variables(host.get_name())
        users = self.getConfiguredUsers(host, host_variables)
        for userName in users:
            self.createHost(f, host, host_variables, False, userName)
        if self.isAnyPortForwarded(host, host_variables):
            self.createHost(f, host, host_variables, True, 'root')

    def writeOutput(self, objectname, f):
        hosts = self.inventory.get_hosts(objectname)
        if hosts is None or len(hosts) == 0:
            raise ValueError("No host matched: " + objectname)
        for host in hosts:
            log("processing host " + host.get_name())
            self.createHosts(f, host)
