# -*- coding: UTF-8 -*-

from ansibleext.inventory_base import InventoryBase


class UvxTestBase(InventoryBase):

    def getDefinedNodesByUvxCluster(self, groupsToInclude, variableToSearch):
        definedNodesByCluster = {}
        for inventory, host in self.inventoriesHostsInAnyGroups(groupsToInclude):
            definedNodeVariable = self.getVariable(inventory, host, variableToSearch)
            clusterServicesGroup = self.getGroupsWithNameMatching(inventory, host, ".+(_services)$", ('logistics_services',))
            cluster = next(iter(clusterServicesGroup))
            if cluster not in definedNodesByCluster:
                definedNodesByCluster[cluster] = []
            if definedNodeVariable:
                definedNodesByCluster[cluster].append(host)
        return definedNodesByCluster

    def verifyExactlyOneItemPerUvxCluster(self, itemsByCluster, type, itemName):
        duplicateItems = {}
        missingItems = {}
        for key in itemsByCluster.keys():
            hosts = itemsByCluster[key]
            numHosts = len(hosts)
            if (numHosts == 0):
                missingItems[key] = hosts
            elif (numHosts > 1):
                duplicateItems[key] = hosts
        self.assertDictEqual(missingItems, {}, f'No {itemName} defined for uvx {type} nodes, but exactly one required: {missingItems}')
        self.assertDictEqual(duplicateItems, {}, f'Multiple {itemName}s defined for {type} nodes, but exactly one required: {duplicateItems}')


class ItrackAndUvxHostsPartOfExactlyOneCluster(UvxTestBase):

    def runTest(self):
        for inventory, host in self.inventoriesHostsInAnyGroups(('itrack', 'uvx')):
            clusterServicesGroups = self.getGroupsWithNameMatching(inventory, host, "^[^_\\s]+(_services)$", ('logistics_services',))
            hostInClustersNum = len(clusterServicesGroups)
            self.assertEqual(hostInClustersNum, 1, '%s must be member of exactly one iTrack cluster (member of exactly 1 <clustername>_services group). Found %s host(s): %s' % (host, hostInClustersNum, clusterServicesGroups))


class UvxOnlyOneCacheTableCreator(UvxTestBase):

    def runTest(self):
        tableCreatorsByCluster = self.getDefinedNodesByUvxCluster(('uvx_cache',), "uvx_tablecreator_node")
        self.verifyExactlyOneItemPerUvxCluster(tableCreatorsByCluster, "cache", "uvx_tablecreator_node")


class UvxOnlyOneCacheCompacter(UvxTestBase):

    def runTest(self):
        cacheCompactersByCluster = self.getDefinedNodesByUvxCluster(('uvx_cache',), "uvx_compacter_node")
        self.verifyExactlyOneItemPerUvxCluster(cacheCompactersByCluster, "cache", "uvx_compacter_node")


class UvxOnlyOneMetadataGc(UvxTestBase):

    def runTest(self):
        metadataGarbageCollectorsByCluster = self.getDefinedNodesByUvxCluster(('uvx_cache',), "uvx_metadatagc_node")
        self.verifyExactlyOneItemPerUvxCluster(metadataGarbageCollectorsByCluster, "cache", "uvx_metadatagc_node")


class UvxOnlyOneReplayJobCleaner(UvxTestBase):

    def runTest(self):
        replayJobCleanersByCluster = self.getDefinedNodesByUvxCluster(('uvx_replay',), "uvx_replayjob_cleaner_node")
        self.verifyExactlyOneItemPerUvxCluster(replayJobCleanersByCluster, "replay", "uvx_replayjob_cleaner_node")


class UvxRequiredVariablesPresent(UvxTestBase):

    def runTest(self):
        for inventory, host in self.inventoriesHostsInAnyGroups(('uvx',)):
            rootVariables = self.getVariablesForHostName(inventory, host.get_name())
            self.assertVariablesExist(host, rootVariables, ['spring_service', 'kafka_config', 'uvx'], 'root')

    def assertVariablesExist(self, host, rootVariable, requiredVariables, variablesPath):
        missingVariables = list(filter(lambda variable: variable not in rootVariable, requiredVariables))
        self.assertEqual(missingVariables, [], "{0} doesn't have all required variables by UVX, missing variables from {1}: {2}".format(host, variablesPath, missingVariables))

