from cassandra.cluster import Cluster  # type: ignore


class CassandraExecutor:
    def __init__(self, ip: str) -> None:
        self.cluster = Cluster(contact_points=[ip], control_connection_timeout=30, connect_timeout=30)
        self.session = None
        print('connecting to host: {}'.format(ip))

    def __enter__(self) -> 'CassandraExecutor':
        assert self.session is None, 'session should not be connected'
        self.session = self.cluster.connect()
        print('connected to host: {}'.format(self.cluster.contact_points))
        return self

    def execute(self, query: str, timeout: int) -> None:
        assert self.session is not None, 'session should be connected'
        self.session.execute(query, timeout=timeout)
        print('executed query: {}'.format(query))

    def __exit__(self, exc_type, exc_val, exc_tb) -> None:
        self.cluster.shutdown()
        print('cluster session and connection closed')


class DryrunCassandraExecutor:
    def __init__(self) -> None:
        pass

    def __enter__(self) -> 'DryrunCassandraExecutor':
        print('created new session')
        return self

    def execute(self, query: str, timeout: int) -> None:
        print('Query to execute: {}'.format(query))

    def __exit__(self, exc_type, exc_val, exc_tb) -> None:
        print('cluster session and connection closed')
