#!/usr/bin/python3
# -*- coding: UTF-8 -*-
'''
idump -- CLI tool to gather and export information about inventories

@copyright:  2015-2020 iData Kft. All rights reserved.
'''

from __future__ import print_function
import sys
import os
import os.path

from argparse import ArgumentParser
from argparse import RawDescriptionHelpFormatter
import traceback

__version__ = 0.3
__date__ = '2017-09-13'
__updated__ = '2017-09-13'

def printerror(*objs):
	print(*objs, file=sys.stderr)

def main(argv=None):
	if argv is None:
		argv = sys.argv
	else:
		sys.argv.extend(argv)

	program_name = os.path.basename(sys.argv[0])
	program_version = "v%s" % __version__
	program_build_date = str(__updated__)
	program_version_message = '%%(prog)s %s (%s)' % (program_version, program_build_date)
	program_shortdesc = __doc__.split("\n")[1]
	program_license = '''%s

	Copyright 2015-2020 iData Kft. All rights reserved.

	Licensed under the GPLv2, same as Ansible.

	Distributed on an "AS IS" basis without warranties
	or conditions of any kind, either express or implied.

USAGE
''' % (program_shortdesc)

	from ansibleext.facade import AnsibleLibraryFacade
	from idump.pac import PACOutputter
	from idump.asbru import AsbruOutputter
	from idump.ssh_config import SSHConfOutputter
	from idump.remmina import RemminaOutputter
	from idump.dumpvars import VarDump
	from idump.dumpvars import ip_status

	try:
		parser = ArgumentParser(description=program_license, formatter_class=RawDescriptionHelpFormatter)
		parser.add_argument('-v', '--verbose', dest='verbose', action="count", help="set verbosity level [default: %(default)s]", default=0)
		parser.add_argument('-V', '--version', action='version', version=program_version_message)
		parser.add_argument('-i', '--inventory', action='append', dest='inventories', help="set inventory file [default: stable inventory, which includes almost all inventory files]")
		parser.add_argument('--var', action='append', dest='vars', help="variable to print, may be specified multiple times. Example: idump --var hostname --var primary_ip nova-main1")
		parser.add_argument('--groups', action='store_true', dest='groups_only', help="Only dump groups of the specified host. Example: idump --groups nova-main1, or with more details: idump --groups -v nova-main1")
		parser.add_argument('--pac', dest='generatePac', help="generate pac import file")
		parser.add_argument('--asbru', dest='generateAsbru', help="generate asbru import file")
		parser.add_argument('--sshconf', dest='generateSSHConf', help="generate basic ssh_conf (no coloring)")
		parser.add_argument('--sshconf-with-coloring', dest='generateSSHConfWithColoring', help="generate ssh_conf file (with coloring)")
		parser.add_argument('--remmina', dest='generateRemmina', help="generate remmina file")
		parser.add_argument('--ipstatus', dest='ipstatus', action='store_true', help="dump free and used ips. Expects a network group, like extended_devnet. Example: idump --ipstatus extended_devnet. Other example: idump --ipstatus ilka_net. Other extended example: idump --ipstatus extended_devnet | grep \"192.168.193.\" | less")
		parser.add_argument('host', help='host or group to print')

		if len(sys.argv) <= 1:
			parser.print_help()
			exit(-1)

		args = parser.parse_args()

		ansible_facade = AnsibleLibraryFacade('.')

		if args.inventories is None:
			args.inventories = ['inventory-stable']
		inventories = [ansible_facade.load_inventory(inventory) for inventory in args.inventories]

		if args.generatePac:
			if len(inventories) != 1:
				raise ValueError("Must specify exactly one inventory!")
			pacOutput = PACOutputter(inventories[0])
			pacOutput.output(args.host, args.generatePac)

		if args.generateAsbru:
			if len(inventories) != 1:
				raise ValueError("Must specify exactly one inventory!")
			AsbruOutput = AsbruOutputter(inventories[0])
			AsbruOutput.output(args.host, args.generateAsbru)

		if args.generateSSHConf:
			SSHConfOutput = SSHConfOutputter(inventories[0], False)
			SSHConfOutput.output(args.host, args.generateSSHConf)

		if args.generateSSHConfWithColoring:
			SSHConfOutput = SSHConfOutputter(inventories[0], True)
			SSHConfOutput.output(args.host, args.generateSSHConfWithColoring)

		if args.generateRemmina:
			RemminaOutput = RemminaOutputter(inventories[0])
			RemminaOutput.output(args.host, args.generateRemmina)

		if args.ipstatus:
			ip_status(inventories[0], args.host)
			return 0

		var_dump = VarDump(filter_vars=args.vars, verbose=args.verbose, groups_only=args.groups_only)
		var_dump.print_trees(inventories=inventories, objectname=args.host)
		return 0
	except KeyboardInterrupt:
		printerror()
		printerror("Interrupted!")
		return 3
	except Exception as e:
		indent = len(program_name) * " "
		printerror(program_name + ": " + str(e) + "\n" + traceback.format_exc() + "\n")
		printerror(indent + "  for help use --help")
		return 2

if __name__ == "__main__":
	sys.exit(main())
