# -*- coding: UTF-8 -*-

import sys
import os
import uuid
import re
import yaml
import ansibleext.yamlsetup as yamlsetup

def log(text):
    print(text)
    sys.stdout.flush()

class PACOutputter:
    def __init__(self, inventory):
        self.inventory = inventory
        self.hosts = dict()
        self.groups = dict()
        self.groupsByName = dict()
        self.ROOT_GROUP_NAME = '__PAC__EXPORTED__'

    def output(self, objectname, pacYml):
        with open(pacYml, "w") as f:
            self.writeOutput(objectname, f)

    def generateUuid(self):
        return uuid.uuid4()

    def getOrError(self, host, dictionary, keys):
        ret = self.getOrDefault(dictionary, keys, None)
        if ret is None:
            raise ValueError('Host ' + host.get_name() + ' has no value for any of ' + str(keys))
        return ret

    def getOrDefault(self, dictionary, keys, default):
        for key in keys:
            if key in dictionary:
                return dictionary[key]
        return default

    def generateName(self, host, portForwardEnabled, userName):
        if portForwardEnabled:
            return host.get_name() + ' portfwd'
        else:
            return host.get_name() + ' (' + userName + ')'

    def generateTitle(self, host, host_variables, portForwardEnabled, userName):
        baseName = self.generateName(host, portForwardEnabled, userName)
        if 'parent' in host_variables:
            return host_variables['parent'] + ' - ' + baseName
        return baseName

    def isStable(self, groups):
        if len(groups) == 0:
            return False
        for group in groups:
            if group.get_name().startswith('stable'):
                return True
            if self.isStable(group.get_parent_groups()):
                return True
        return False

    def generateColor(self, host, host_variables):
        if self.isStable(host.get_groups()):
            return '#550000000000'
        else:
            return '#000000000000'

    def addLocalPortForward(self, options, port):
        options.extend(['-L', 'localhost/' + str(port) + '/localhost/' + str(port)])

    def getOrWarning(self, host, mapVar, keySequence):
        current = mapVar
        current_key = []
        for key in keySequence:
            if not isinstance(current, dict):
                log('Warning: key ' + '.'.join(current_key) + ' required to access path ' + '.'.join(keySequence) + ' not a map for host ' + host.get_name())
                raise ValueError
            current_key.append(key)
            if not key in current:
                log('Warning: key ' + '.'.join(current_key) + ' required to access path ' + '.'.join(keySequence) + ' not found for host ' + host.get_name())
                raise ValueError
            current = current[key]
        return current

    def addItrackPortForward(self, options, host, host_variables):
        try:
            rmi_port = self.getOrWarning(host, host_variables, ['itrack_config', 'jmx', 'rmi_port'])
            rmiregistry_port = self.getOrWarning(host, host_variables, ['itrack_config', 'jmx', 'rmiregistry_port'])
            self.addLocalPortForward(options, rmi_port)
            self.addLocalPortForward(options, rmiregistry_port)
        except ValueError:
            pass
        self.addLocalPortForward(options, 8000)

    def isAnyPortForwarded(self, host, host_variables):
        options = self.getSshOptions(host, host_variables, True)
        return '-L' in options

    def getSshOptions(self, host, host_variables, portForwardEnabled):
        options = ['-2', '-x', '-A']
        if portForwardEnabled:
            if 'itrack_config' in host_variables:
                self.addItrackPortForward(options, host, host_variables)
        return options

    def generateSshOptions(self, host, host_variables, portForwardEnabled):
        options = self.getSshOptions(host, host_variables, portForwardEnabled)
        return ' ' + ' '.join(options)

    def getConfiguredUsers(self, host, host_variables):
        users = ['root']
        if 'authpersonmap' in host_variables:
            rawlist = host_variables['authpersonmap']
            if rawlist is not None:
                personmap = rawlist.values()
                splitted_personmap = []
                for x in personmap:
                    splitted_personmap = splitted_personmap + x.split(",")
                    if len(personmap) != 0:
                        tmp = users + splitted_personmap
                        users = list(dict.fromkeys(tmp))
        return users

    def createHost(self, host, host_variables, portForwardEnabled, userName):
        title = self.generateTitle(host, host_variables, portForwardEnabled, userName)
        ret = dict()
        ret['_is_group'] = 0
        ret['_protected'] = 0
        ret['KPX title regexp'] = '.*' + title + '.*'
        ret['auth fallback'] = 1
        ret['auth type'] = 'publickey'
        ret['autoreconnect'] = ''
        ret['autossh'] = ''
        ret['children'] = dict()
        ret['cluster'] = []
        ret['description'] = 'Connection with ' + host.get_name() + ' as user ' + userName
        ret['embed'] = 0
        ret['expect'] = []
        ret['favourite'] = 0
        ret['infer from KPX where'] = 3
        ret['infer user pass from KPX'] = ''
        ret['ip'] = self.getOrDefault(host_variables, ['ansible_host'], host.get_name())
        ret['local after'] = []
        ret['local before'] = []
        ret['local connected'] = []
        ret['mac'] = ''
        ret['macros'] = []
        ret['method'] = 'SSH'
        ret['name'] = self.generateName(host, portForwardEnabled, userName) 
        ret['options'] = self.generateSshOptions(host, host_variables, portForwardEnabled)
        ret['parent'] = None
        ret['pass'] = ''
        ret['passphrase'] = ''
        ret['passphrase user'] = userName
        ret['port'] = self.getOrDefault(host_variables, ['ansible_port'], 22)
        ret['prepend command'] = ''
        ret['proxy ip'] = ''
        ret['proxy pass'] = ''
        ret['proxy port'] = 8080
        ret['proxy user'] = ''
        ret['public key'] = os.path.expanduser('~') + '/.ssh/id_rsa'
        ret['quote command'] = ''
        ret['remove control chars'] = ''
        ret['save session logs'] = ''
#        ret['screenshots'] = None
        ret['search pass on KPX'] = 0
        ret['send slow'] = 0
        ret['send string active'] = ''
        ret['send string every'] = 60
        ret['send string intro'] = 1
        ret['send string txt'] = ''
        ret['session log pattern'] = '<UUID>_<NAME>_<DATE_Y><DATE_M><DATE_D>_<TIME_H><TIME_M><TIME_S>.txt'
        ret['session logs amount'] = 10
        ret['startup launch'] = ''
        ret['startup script'] = ''
        ret['startup script name'] = ''
        terminal_options = dict()
        ret['terminal options'] = terminal_options
        terminal_options['audible bell'] = ''
        terminal_options['back color'] = self.generateColor(host, host_variables)
        terminal_options['bold color'] = '#cc62cc62cc62'
        terminal_options['bold color like text'] = 1
        terminal_options['command prompt'] = r'[#%\$>]|\:\/\s*$'
        terminal_options['cursor shape'] = 'block'
        terminal_options['disable ALT key bindings'] = ''
        terminal_options['disable CTRL key bindings'] = ''
        terminal_options['disable SHIFT key bindings'] = ''
        terminal_options['open in tab'] = 1
        terminal_options['password prompt'] = "([p|P]ass|[p|P]ass[w|W]or[d|t]|Enter passphrase for key '.+'):\\s*$"
        terminal_options['tab back color'] = '#000000000000'
        terminal_options['terminal backspace'] = 'auto'
        terminal_options['terminal character encoding'] = 'UTF-8'
        terminal_options['terminal emulation'] = 'xterm'
        terminal_options['terminal font'] = 'Monospace 9'
        terminal_options['terminal scrollback lines'] = '5000'
        terminal_options['terminal transparency'] = 0
        terminal_options['terminal window hsize'] = 800
        terminal_options['terminal window vsize'] = 600
        terminal_options['text color'] = '#cc62cc62cc62'
        terminal_options['timeout command'] = 40
        terminal_options['timeout connect'] = 40
        terminal_options['use personal settings'] = 1
        terminal_options['use tab back color'] = ''
        terminal_options['username prompt'] = r'([l|L]ogin|[u|u]suario|[u|U]ser-?[n|N]ame|[u|U]ser):\s*$'
        terminal_options['visible bell'] = ''
        ret['title'] = title
        ret['use prepend command'] = ''
        ret['use proxy'] = 0
        ret['user'] = ''
        ret['variables'] = []
        uuid = str(self.generateUuid())
        self.hosts[uuid] = ret
        return uuid

    def createGroup(self, group, path):
        ret = dict()
        ret['_is_group'] = 1
        ret['_protected'] = 0
        ret['description'] = 'Connection group ' + path
        ret['expect'] = []
        ret['name'] = group.get_name()
        ret['parent'] = None
#        ret['screenshots'] = None
        ret['children'] = dict()
        uuid = str(self.generateUuid())
        self.groups[uuid] = ret
        self.groupsByName[path] = uuid
        return uuid

    def transformPaths(self, paths):
        unique = dict()
        for path in paths:
            if path[0] in ['all', 'idata']:
                path[0] = 'all'
            pathName = '.'.join(path)
            if pathName not in unique:
                unique[pathName] = path
        return unique.values()

    def getUniquePaths(self, groups):
        pathsAccumulated = []
        self.getUniquePathsImpl(groups, [], pathsAccumulated)
        return self.transformPaths(pathsAccumulated)

    def matchesPathWithRelevantGroupingInformation(self, path):
        fullPath = '.'.join(path)
        ansible_version = self.inventory.get_ansible_version()
        if ansible_version.startswith('1.'):
            return re.match(r'all\.[^.]+_(net|cassandra|itrack|virtual|hwn|firewall|tomcat|openvz|dispatcher)', fullPath) is not None
        else:
            # eliminate redundant groupings (at least for PAC)
            if len(path) > 2:
                if path[0] == 'all':
                    if path[1] == 'idata':
                        return False
            return re.match(r'all\..*_(net|cassandra|itrack|virtual|hwn|firewall|tomcat|openvz|dispatcher)', fullPath) is not None

    def getUniquePathsImpl(self, groups, currentPath, pathsAccumulated):
        if len(currentPath) > 0 and self.matchesPathWithRelevantGroupingInformation(currentPath):
            pathsAccumulated.append(currentPath)
        for group in groups:
            self.getUniquePathsImpl(group.get_parent_groups(), [group.get_name()] + currentPath, pathsAccumulated)

    def getOrCreateGroup(self, group, groupPath):
        if group.get_name() in ['all']:
            return self.groupsByName[self.ROOT_GROUP_NAME]
        if groupPath in self.groupsByName:
            return self.groupsByName[groupPath]
        else:
            return self.createGroup(group, groupPath)

    def createRequiredGroupsForPath(self, path):
        parentGroupUuid = None
        for prefixLen in range(1, len(path)+1):
            prefix = path[0:prefixLen]
            group = self.inventory.get_group(prefix[-1])
            assert group is not None
            groupPath = '.'.join(prefix)
            groupUuid = self.getOrCreateGroup(group, groupPath)
            if parentGroupUuid is not None:
                self.addChildToGroup(parentGroupUuid, groupUuid)
            parentGroupUuid = groupUuid
        return parentGroupUuid

    def getEntryByUuid(self, uuid):
        if uuid in self.hosts:
            return self.hosts[uuid]
        if uuid in self.groups:
            return self.groups[uuid]
        raise ValueError()

    def addChildToGroup(self, groupUuid, childUuid):
        parent = self.getEntryByUuid(groupUuid)
        child = self.getEntryByUuid(childUuid)
        if child['parent'] == groupUuid:
            return
        assert child['parent'] is None, "node " + childUuid + " already has a parent"
        parent['children'][childUuid] = 1
        child['parent'] = groupUuid

    def createRoot(self):
        ret = dict()
        ret['children'] = dict()
        self.groups[self.ROOT_GROUP_NAME] = ret
        self.groupsByName[self.ROOT_GROUP_NAME] = self.ROOT_GROUP_NAME

    def writeOutput(self, objectname, f):
        hosts = self.inventory.get_hosts(objectname)
        if hosts is None or len(hosts) == 0:
            raise ValueError("No host matched: " + objectname)
        self.createRoot()
        for host in hosts:
            log("processing host " + host.get_name())
            host_variables = self.inventory.get_variables(host.get_name())
            users = self.getConfiguredUsers(host, host_variables)
            forwarding = self.isAnyPortForwarded(host, host_variables)
            uniquePaths = self.getUniquePaths(host.get_groups())
            if uniquePaths == []:
                log("no unique paths for " + host.get_name())
            for path in uniquePaths:
                groupUuid = self.createRequiredGroupsForPath(path)
                for user in users:
                    hostUuid = self.createHost(host, host_variables, False, user)
                    self.addChildToGroup(groupUuid, hostUuid)
                if forwarding:
                    hostUuid = self.createHost(host, host_variables, True, 'root')
                    self.addChildToGroup(groupUuid, hostUuid)

        f.write("---\n")
        yamlsetup.add_ansible_representers()
        f.write(yaml.safe_dump(self.hosts, default_flow_style=False, default_style=None, line_break=False, width=1000000))
        f.write(yaml.safe_dump(self.groups, default_flow_style=False, default_style=None, line_break=False, width=1000000))
