# -*- coding: UTF-8 -*-

import os
import subprocess
import tests.test_library as testlib
from typing import List

import sys
sys.path.append('playbooks/library')
import git_tool

class TestGitToolModule(testlib.TestModuleBase):

    def setUp(self) -> None:
        testlib.TestModuleBase.setUp(self)
        (self.tmpdir) = self.setup_tmp_dir()
        self.repo_path = os.path.join(self.tmpdir, 'test_repo')
        self.repo_path_2 = os.path.join(self.tmpdir, 'test_repo_2')
        self.hook_path = self._hook_file('hook_script')

    def _hook_file(self, name: str) -> str:
        hook_path = os.path.join(self.tmpdir, name)
        with open(hook_path, 'w') as f:
            f.write("#!/bin/sh\n")
        os.chmod(hook_path, 0o700)
        return hook_path

    def assertRunYieldsNoChanges(self, args: dict) -> None:
        with self.assertRaises(testlib.AnsibleExitJson) as result:
            self.set_module_args(args)
            git_tool.main()
        json = result.exception.args[0]
        self.assertIn('changed', json)
        self.assertEqual(json['changed'], False)
        if 'diff' in json:
            self.assertDictEqual(json['diff'], {
                'after': None,
                'before': None,
            })

    def assertFileIsSymlinkTo(self, symlink_file: str, dest_file: str) -> None:
        hook_path = os.path.join(self.tmpdir, dest_file)
        self.assertTrue(os.path.islink(symlink_file), f'{symlink_file} must be a symlink')
        self.assertEqual(os.readlink(symlink_file), hook_path, f'{symlink_file} must point to {hook_path}')

    def runGit(self, args: List[str]) -> None:
        process = subprocess.run(['git'] + args, stdout=subprocess.PIPE, stderr=subprocess.PIPE, check=False)
        if process.returncode != 0:
            print(f"Error calling git {args}:\n* stdout={process.stdout!r}\n* stderr={process.stderr!r}")
        process.check_returncode()

    def create_empty_repo(self, repo_path: str) -> None:
        os.mkdir(repo_path)
        with self.in_dir(repo_path):
            self.runGit(['init'])
            self.runGit(['config', '--local', 'user.name', 'Test User'])
            self.runGit(['config', '--local', 'user.email', 'testuser@example.com'])
            self.runGit(['config', '--local', 'commit.gpgsign', 'false'])
            self.write_file('testfile', 'content')
            self.runGit(['add', 'testfile'])
            self.runGit(['commit', '-m', 'message'])

    def test_module_fail_when_required_args_missing(self):
        with self.assertRaises(testlib.AnsibleFailJson):
            self.set_module_args({})
            git_tool.main()

    def test_git_clone_with_existing_directory(self):
        self.create_empty_repo(self.repo_path_2)
        os.mkdir(self.repo_path)
        args = {
            'repo': self.repo_path,
            'state': 'checked_out',
            'clone': self.repo_path_2,
            'refname': 'master',
        }
        expected = {
            'changed': True,
        }
        with self.assertRaises(testlib.AnsibleExitJson) as result:
            self.set_module_args(args)
            git_tool.main()

        json = result.exception.args[0]
        self.assertDictEqual(json, expected)
        self.assertTrue(os.path.isdir(os.path.join(self.repo_path, '.git')))
        self.assertRunYieldsNoChanges(args)

    def test_git_clone_without_existing_directory(self):
        self.create_empty_repo(self.repo_path_2)
        args = {
            'repo': self.repo_path,
            'state': 'checked_out',
            'clone': self.repo_path_2,
            'refname': 'master',
        }
        expected = {
            'changed': True,
        }
        with self.assertRaises(testlib.AnsibleExitJson) as result:
            self.set_module_args(args)
            git_tool.main()

        json = result.exception.args[0]
        self.assertDictEqual(json, expected)
        self.assertTrue(os.path.isdir(self.repo_path))
        self.assertTrue(os.path.isdir(os.path.join(self.repo_path, '.git')))
        self.assertRunYieldsNoChanges(args)

    def test_git_commit_all(self):
        self.create_empty_repo(self.repo_path_2)
        self.runGit(['clone', self.repo_path_2, self.repo_path])
        with self.in_dir(self.repo_path):
            self.runGit(['config', '--local', 'commit.gpgsign', 'false'])
            self.write_file('newfile1', 'newfile1_content')
            self.write_file('newfile2', 'newfile2_content')

        args = {
            'repo': self.repo_path,
            'state': 'committed',
            'paths': '--all',
            'user': 'Erős Pista <pista.eros@idata.hu>',
            'commit_message': 'test commit'
        }
        expected = {
            'changed': True,
        }
        with self.assertRaises(testlib.AnsibleExitJson) as result:
            self.set_module_args(args)
            git_tool.main()

        json = result.exception.args[0]
        self.assertDictEqual(json, expected)
        self.assertRunYieldsNoChanges(args)

    def test_init_bare_and_config(self):
        args = {
            'repo': self.repo_path,
            'state': 'config',
            'merge': False,
            'init': [
                '--bare'
            ],
            'git_config': {
                'some.config1': 'aaa',
                'some.config2': 'bbb',
                'some.listconfig': ['item1', 'item2']
            },
            'hook_config': {
                'post-receive': self._hook_file('hook_script1'),
                'post-receive.d': [
                    self._hook_file('hook_script.d_1'),
                    self._hook_file('hook_script.d_2')
                ],
                'update': self._hook_file('hook_script2')
            },
        }
        expected = {
            'changed': True,
            'changed_git_hooks': ['update', 'post-receive', 'post-receive.d'],
            'diff' : {
               'prepared': 'Changed hooks:\n'
                        ' - update\n'
                        ' - post-receive\n'
                        ' - post-receive.d',
                'before': None,
                'after': "- 'some.config1: aaa'\n"
                         "- 'some.config2: bbb'\n"
                         "- 'some.listconfig: item1'\n"
                         "- 'some.listconfig: item2'\n"
             }
        }
        with self.assertRaises(testlib.AnsibleExitJson) as result:
            self.set_module_args(args)
            git_tool.main()

        json = result.exception.args[0]
        self.assertDictEqual(json, expected)
        self.assertFileIsSymlinkTo(os.path.join(self.repo_path, 'hooks', 'post-receive'), 'hook_script1')
        self.assertFileIsSymlinkTo(os.path.join(self.repo_path, 'hooks', 'post-receive.d', 'hook_script.d_1'), 'hook_script.d_1')
        self.assertFileIsSymlinkTo(os.path.join(self.repo_path, 'hooks', 'post-receive.d', 'hook_script.d_2'), 'hook_script.d_2')
        self.assertFileIsSymlinkTo(os.path.join(self.repo_path, 'hooks', 'update'), 'hook_script2')
        self.assertRunYieldsNoChanges(args)
