# -*- coding: UTF-8 -*-

'''
Facade over Ansible's inventory, to be able to support various versions of the library at once.
'''

import os
import copy

from ansible.parsing.yaml.objects import AnsibleUnicode # type: ignore
from datetime import date

from typing import List, Optional, Dict, Any

def recursive_ununicode(node, ansible):
    if node is None:
        return None
    if isinstance(node, AnsibleUnicode):
        return str(node)
    if isinstance(node, str):
        return node
    if isinstance(node, int):
        return node
    if isinstance(node, float):
        return node
    if isinstance(node, bool):
        return node
    if isinstance(node, date):
        return node
    if isinstance(node, list):
        return [recursive_ununicode(elem, ansible) for elem in node]
    if isinstance(node, dict):
        return {recursive_ununicode(k, ansible): recursive_ununicode(v, ansible) for k, v in node.items()}
    if isinstance(node, ansible.parsing.yaml.objects.AnsibleVaultEncryptedUnicode):
        return node._ciphertext
    raise ValueError(f'unknown node: {type(node)}')

class GroupFacade:

    __slots__ = [ '_group' ]

    def __init__(self, group):
        assert group is not None
        self._group = group

    def __repr__(self) -> str:
        return 'Group ' + self.get_name()

    def get_name(self) -> str:
        return self._group.name

    def get_depth(self) -> int:
        return self._group.depth

    def get_hosts(self) -> List['HostFacade']:
        return [HostFacade(host) for host in self._group.hosts]

    def get_parent_groups(self) -> List['GroupFacade']:
        return [GroupFacade(group) for group in self._group.parent_groups]

    def get_ancestor_groups(self) -> List['GroupFacade']:
        return [GroupFacade(group) for group in self._group.get_ancestors()]

    def get_child_groups(self) -> List['GroupFacade']:
        return [GroupFacade(group) for group in self._group.child_groups]

class HostFacade:

    __slots__ = [ '_host' ]

    def __init__(self, host):
        assert host is not None
        self._host = host

    def __repr__(self) -> str:
        return 'Host ' + self.get_name()

    def get_name(self) -> str:
        return self._host.name

    def get_groups(self) -> List[GroupFacade]:
        return [GroupFacade(group) for group in self._host.groups]

class InventoryFacadeBase:
    def __init__(self, inventory, _ansible_facade: 'AnsibleLibraryFacade'):
        self._inventory = inventory
        self._ansible_facade = _ansible_facade

    def get_ansible_version(self) -> str:
        return self._ansible_facade.version

    def get_group(self, name: str) -> GroupFacade:
        group = self.get_group_or_none(name)
        if group is None:
            raise ValueError(f'No such group: {name}')
        return group

    def get_group_or_none(self, name: str) -> Optional[GroupFacade]:
        group = self._inventory.get_group(name)
        if group is None:
            return None
        return GroupFacade(group)

    def groups_for_host(self, host: str) -> List[GroupFacade]:
        return [GroupFacade(group) for group in self._inventory.groups_for_host(host)]

    def get_host(self, name: str) -> HostFacade:
        host = self.get_host_or_none(name)
        if host is None:
            raise ValueError(f'No such host: {name}')
        return host

    def get_host_or_none(self, name: str) -> Optional[HostFacade]:
        host = self._inventory.get_host(name)
        if host is None:
            return None
        return HostFacade(host)

    def get_hosts(self, pattern: str) -> List[HostFacade]:
        return [HostFacade(host) for host in self._inventory.get_hosts(pattern)]

    def get_variables(self, hostname: str) -> Dict[str, Any]:
        raise NotImplementedError()

    def create_host(self, hostname: str, group: GroupFacade) -> HostFacade:
        raise NotImplementedError()

class InventoryFacade_v2_1X(InventoryFacadeBase):
    def __init__(self, inventory, variable_manager, ansible_facade: 'AnsibleLibraryFacade'):
        InventoryFacadeBase.__init__(self, inventory, ansible_facade)
        self._variable_manager = variable_manager
        self._variable_cache: Dict[str, Any] = {}

    def __repr__(self) -> str:
        return 'Inventory ' + repr(self._inventory._sources)

    def get_name(self) -> str:
        return repr(self._inventory._sources)

    def groups_for_host(self, host: str) -> List[GroupFacade]:
        return [GroupFacade(group) for group in self._groups_for_host(host)]

    def _groups_for_host(self, pattern: str):
        hosts = self._inventory.get_hosts(pattern)
        if len(hosts) == 0:
            return []
        assert len(hosts) == 1
        return hosts[0].get_groups()

    def get_group_or_none(self, name: str) -> Optional[GroupFacade]:
        group = self._inventory.groups.get(name)
        if group is None:
            return None
        return GroupFacade(group)

    def get_variables(self, hostname: str) -> Dict[str, Any]:
        if hostname not in self._variable_cache:
            self._variable_cache[hostname] = self._get_variables_uncached(hostname)
        return copy.copy(self._variable_cache[hostname])

    def create_host(self, hostname: str, group: GroupFacade) -> HostFacade:
        assert isinstance(group, GroupFacade)
        self._inventory.add_host(host=hostname, group=group._group.get_name())
        self._inventory.reconcile_inventory()
        new_host = self.get_host_or_none(hostname)
        assert new_host is not None
        return new_host

    def _get_variables_uncached(self, hostname: str) -> Any:
        assert isinstance(hostname, str)
        host = self._inventory.get_host(hostname)
        if host is None:
            raise ValueError('No such host: ' + hostname)
        variables = self._variable_manager.get_vars(host=host)
        return recursive_ununicode(variables, self._ansible_facade._ansible)

"""
Facade over Ansible library, to be able to support various versions of the library at once.
"""
class AnsibleLibraryFacade:
    def __init__(self, root):
        self._root = root
        self._ansible = self.detect_and_load_ansible_library()

    def load_ansible_library_common(self):
        # this environment variable is used by ansible to find the current configuration correctly (needed for hash_behaviour)
        os.environ['ANSIBLE_CONFIG'] = os.path.join(os.path.dirname(self._root), 'ansible.cfg')

        # its current value may be debugged by:
        # import ansible.constants as C
        # printerror(C.DEFAULT_HASH_BEHAVIOUR)

        # only import ansible after environment has been set up
        import ansible.errors # type: ignore
        import ansible.parsing.dataloader # type: ignore
        import ansible.parsing.yaml # type: ignore
        return ansible # type: ignore

    def load_ansible_library_2_1X(self):
        ansible = self.load_ansible_library_common()
        import ansible.vars.manager # type: ignore
        import ansible.inventory.manager # type: ignore
        import ansible.parsing.yaml.objects # type: ignore
        self.inventory_impl_version = '2.1X'
        self.AnsibleVaultEncryptedUnicode_data_monkey_patching(ansible)
        return ansible # type: ignore

    def detect_and_load_ansible_library(self):
        import ansible # type: ignore
        self.version = ansible.__version__
        if self.version.startswith(('2.17.')):
            return self.load_ansible_library_2_1X()
        else:
            raise ValueError('Ansible-core version ' + self.version + ' not supported')

    def load_inventory(self, inventory_path: str) -> InventoryFacadeBase:
        if self.inventory_impl_version == '2.1X':
            loader = self._ansible.parsing.dataloader.DataLoader() # type: ignore
            native_inventory = self._ansible.inventory.manager.InventoryManager(loader=loader, sources=inventory_path)
            variable_manager = self._ansible.vars.manager.VariableManager(loader=loader, inventory=native_inventory)
            return InventoryFacade_v2_1X(native_inventory, variable_manager, self)
        else:
            raise ValueError('Ansible-core version ' + self.version + ' not supported')

    # Fix 'AnsibleVaultError: Attempting to decrypt but no vault secrets found' on 'idump -i inventory-stable "*"' command.
    def AnsibleVaultEncryptedUnicode_data_monkey_patching(self, ansible):
        from ansible.module_utils.common.text.converters import to_text  # type: ignore

        def data_without_vault_decrypt(self):
            return to_text(self._ciphertext)

        if self.version.startswith('2.17.'):
            ansible.parsing.yaml.objects.AnsibleVaultEncryptedUnicode.data = property(data_without_vault_decrypt)
