#!/usr/bin/python3

import argparse
from idata_cassandra import cassandra_executor  # type: ignore
from ansibleext.facade import AnsibleLibraryFacade  # type: ignore
from datetime import datetime
from typing import List,Generator


def generate_date_postfixes(compactcache_months: List[int]) -> Generator[str, None, None]:
    current_date = datetime.today()
    current_month = current_date.month
    current_year = current_date.year
    for month in compactcache_months:
        if current_month < month:
            yield '{}{:0>2d}'.format(current_year - 1, month)
        else:
            yield '{}{:0>2d}'.format(current_year, month)
    whole_year = False
    for month in compactcache_months:
        if current_month == month - 1:
            whole_year = True
    if not whole_year:
        for month in compactcache_months:
            if abs(current_month - month) < 2 and current_month <= month:
                yield '{}{:0>2d}'.format(current_year - 1, month)


def generate_cqlsh_truncate_statements(keyspace: str, compactcache_months: List[int]) -> Generator[str, None, None]:
    for date in generate_date_postfixes(compactcache_months):
        yield 'TRUNCATE {}.compactcache_{};'.format(keyspace, date)
    yield 'TRUNCATE {}.tailcache;'.format(keyspace)


def set_default_args_values_if_needed(args: argparse.Namespace, default_timeout: int) -> None:
    if args.timeout is None:
        args.timeout = default_timeout


def parse_args(default_timeout: int) -> argparse.Namespace:
    parser = argparse.ArgumentParser(description='Truncates given cassandra tables on the given host using the given '
                                                 'keyspace(s).')
    parser.add_argument('--hostname', '-H', type=str, required=True,
                        help='The hostname of the cassandra to use for running the commands.')
    parser.add_argument('--keyspaces', '-k', type=str, required=True, nargs='+',
                        help='The requested cassandra keyspace(s).')
    parser.add_argument('--timeout', '-t', type=int, help='The truncate --request-timeout. Default '
                                                          'value: 60')
    parser.add_argument('--dry-run', help='No-op, do not change anything. Print all the commands, but do not execute '
                                          'them', action='store_true')
    args = parser.parse_args()
    set_default_args_values_if_needed(args, default_timeout)
    return args


def get_executor(args: argparse.Namespace):
    print('checking requested host in inventory-testing')
    ansible_library_facade = AnsibleLibraryFacade('infrastructure')
    host = ansible_library_facade.load_inventory('inventory-testing').get_variables(args.hostname)
    ip = host['primary_ip']
    print('host found, the primary ip is {}'.format(ip))
    if args.dry_run:
        return cassandra_executor.DryrunCassandraExecutor()
    else:
        return cassandra_executor.CassandraExecutor(ip)


def main():
    compactcache_months = [1, 4, 7, 10]
    default_timeout = 60
    args = parse_args(default_timeout)
    keyspaces = args.keyspaces
    with get_executor(args) as executor:
        for current_keyspace in keyspaces:
            for statement in generate_cqlsh_truncate_statements(current_keyspace, compactcache_months):
                executor.execute(statement, args.timeout)


main()
