# -*- coding: UTF-8 -*-

import sys
import os
import urllib

def log(text):
    print(text)
    sys.stdout.flush()

class RemminaOutputter:
    def __init__(self, inventory):
        self.inventory = inventory
        self.hosts = dict()
        self.groups = dict()
        self.groupsByName = dict()

    def output(self, objectname, outputDirectory):
        self.writeOutput(objectname, outputDirectory)

    def getOrError(self, host, dictionary, keys):
        ret = self.getOrDefault(dictionary, keys, None)
        if ret is None:
            raise ValueError('Host ' + host.get_name() + ' has no value for any of ' + str(keys))
        return ret

    def getOrDefault(self, dictionary, keys, default):
        for key in keys:
            if key in dictionary:
                return dictionary[key]
        return default

    def generateName(self, host, portForwardEnabled, userName):
        if portForwardEnabled:
            return host.get_name() + ' portfwd'
        else:
            return host.get_name() + ' (' + userName + ')'

    def isStable(self, groups):
        if len(groups) == 0:
            return False
        for group in groups:
            if group.get_name().startswith('stable'):
                return True
            if self.isStable(group.get_parent_groups()):
                return True
        return False
    
    def isDevel(self, groups):
        if len(groups) == 0:
            return False
        for group in groups:
            if group.get_name().startswith('devel'):
                return True
            if self.isDevel(group.get_parent_groups()):
                return True
        return False

    def generateColor(self, host, host_variables):
        if self.isStable(host.get_groups()):
            return 'iDataStable'
        else:
            return '0'
    
    def generateGroup(self, host, host_variables):
        if self.isStable(host.get_groups()):
            return 'stable'
        elif self.isDevel(host.get_groups()):
            return 'devel'
        else:
            return 'testing'

    def addLocalPortForward(self, forwards, port):
        forwards.append('L' + str(port) + '=localhost:' + str(port))

    def getOrWarning(self, host, mapVar, keySequence):
        current = mapVar
        current_key = []
        for key in keySequence:
            if not isinstance(current, dict):
                log('Warning: key ' + '.'.join(current_key) + ' required to access path ' + '.'.join(keySequence) + ' not a map for host ' + host.get_name())
                raise ValueError
            current_key.append(key)
            if not key in current:
                log('Warning: key ' + '.'.join(current_key) + ' required to access path ' + '.'.join(keySequence) + ' not found for host ' + host.get_name())
                raise ValueError
            current = current[key]
        return current

    def addItrackPortForward(self, forwards, host, host_variables):
        try:
            rmi_port = self.getOrWarning(host, host_variables, ['itrack_config', 'jmx', 'rmi_port'])
            rmiregistry_port = self.getOrWarning(host, host_variables, ['itrack_config', 'jmx', 'rmiregistry_port'])
            self.addLocalPortForward(forwards, rmi_port)
            self.addLocalPortForward(forwards, rmiregistry_port)
        except ValueError:
            pass
        self.addLocalPortForward(forwards, 8000)

    def isAnyPortForwarded(self, host, host_variables):
        forwards = self.generatePortforwards(host, host_variables, True)
        print(str(not not forwards) + 'for: ' + forwards)
        return not not forwards

    def generatePortforwards(self, host, host_variables, portForwardEnabled):
        forwards = []
        if portForwardEnabled:
            if 'itrack_config' in host_variables:
                self.addItrackPortForward(forwards, host, host_variables)
        return ','.join(forwards)

    def getConfiguredUsers(self, host, host_variables):
        users = ['root']
        if 'authpersonmap' in host_variables:
            rawlist = host_variables['authpersonmap']
            if rawlist is not None:
                personmap = rawlist.values()
                splitted_personmap = []
                for x in personmap:
                    splitted_personmap = splitted_personmap + x.split(",")
                    if len(personmap) != 0:
                        tmp = users + splitted_personmap
                        users = list(dict.fromkeys(tmp))
        return users

    def createHost(self, host, host_variables, portForwardEnabled, userName):
        ip = self.getOrDefault(host_variables, ['ansible_host'], host.get_name())
        port = self.getOrDefault(host_variables, ['ansible_port'], 22)
        user = userName
        portForwards = self.generatePortforwards(host, host_variables, portForwardEnabled)
        backgroundColor = self.generateColor(host, host_variables)
        group = self.generateGroup(host, host_variables)
        name = self.generateName(host, portForwardEnabled, userName)

        content = ''\
        '[remmina]\n'\
        'ssh_auth=3\n'\
        'ssh_password=\n'\
        'postcommand=\n'\
        'ssh_privatekey=' + os.path.expanduser('~') + '/.ssh/id_rsa.pub\n'\
        'ssh_color_scheme=' + backgroundColor + '\n'\
        'ssh_passphrase=\n'\
        'ssh_hostkeytypes=\n'\
        'ssh_loopback=0\n'\
        'group=' + group + '\n'\
        'name=' + name + '\n'\
        'precommand=\n'\
        'ssh_charset=\n'\
        'ssh_username=' + user + '\n'\
        'ssh_server=\n'\
        'ssh_compression=0\n'\
        'protocol=SSH\n'\
        'ssh_ciphers=\n'\
        'sshlogenabled=0\n'\
        'exec=\n'\
        'ssh_enabled=0\n'\
        'ssh_stricthostkeycheck=0\n'\
        'ssh_proxycommand=\n'\
        'disablepasswordstoring=0\n'\
        'sshlogname=\n'\
        'server=' + ip + ':' + str(port) + '\n'\
        'sshlogfolder=\n'\
        'ssh_kex_algorithms=\n'\
        'save_ssh_server=\n'\
        'save_ssh_username=\n'\
        'last_success=19700101\n'\
        'window_maximize=0\n'\
        'viewmode=1\n'\
        'window_width=672\n'\
        'window_height=506\n'
        return content

    def createFilenameNoForwarding(self, host, user):
        return urllib.parse.quote('idata-' + host.get_name() + '-' + user + '.remmina' + '', '')

    def createFilenameForwarding(self, host):
        return urllib.parse.quote('idata-' + host.get_name() + '-portfwd' + '.remmina' , '')

    def writeOutput(self, objectname, outputDirectory):
        hosts = self.inventory.get_hosts(objectname)
        if hosts is None or len(hosts) == 0:
            raise ValueError("No host matched: " + objectname)
        for host in hosts:
            log("processing host " + host.get_name())
            host_variables = self.inventory.get_variables(host.get_name())
            users = self.getConfiguredUsers(host, host_variables)
            forwarding = self.isAnyPortForwarded(host, host_variables)
            for user in users:
                content = self.createHost(host, host_variables, False, user)
                filename = self.createFilenameNoForwarding(host, user)
                with open(os.path.join(outputDirectory, filename), "w") as f:
                    f.write(content)
            if forwarding:
                content = self.createHost(host, host_variables, True, 'root')
                filename = self.createFilenameForwarding(host)
                with open(os.path.join(outputDirectory, filename), "w") as f:
                    f.write(content)
