import unittest
import socket
from unittest.mock import patch
from jinja2.exceptions import TemplateError

import playbooks.filter_plugins.ipv4 as ipv4

class CidrToSameIp(unittest.TestCase):
    def test_happy(self):
        assert ipv4.cidr_to_same_ip('1.2.3.4/0') == '1.2.3.4'
        assert ipv4.cidr_to_same_ip('1.2.3.4/1') == '1.2.3.4'
        assert ipv4.cidr_to_same_ip('1.2.3.4/32') == '1.2.3.4'

    def test_non_ipv4(self):
        with self.assertRaises(ValueError):
            ipv4.cidr_to_same_ip('1.2.3.555')
        with self.assertRaises(ValueError):
            ipv4.cidr_to_same_ip('1.2.3.4/33')
        with self.assertRaises(ValueError):
            ipv4.cidr_to_same_ip('example.com')


class CidrToNetwork(unittest.TestCase):
    def test_happy(self):
        assert ipv4.cidr_to_network('1.2.3.4/0') == '0.0.0.0'
        assert ipv4.cidr_to_network('129.2.3.4/1') == '128.0.0.0'
        assert ipv4.cidr_to_network('254.2.3.4/2') == '192.0.0.0'
        assert ipv4.cidr_to_network('1.2.3.255/31') == '1.2.3.254'
        assert ipv4.cidr_to_network('1.2.3.4/32') == '1.2.3.4'

    def test_non_ipv4(self):
        with self.assertRaises(ValueError):
            ipv4.cidr_to_network('1.2.3.555')
        with self.assertRaises(ValueError):
            ipv4.cidr_to_network('1.2.3.4/33')
        with self.assertRaises(ValueError):
            ipv4.cidr_to_network('1.2.3.4/-1')
        with self.assertRaises(ValueError):
            ipv4.cidr_to_network('example.com')


class CidrToNetmask(unittest.TestCase):
    def test_happy(self):
        assert ipv4.cidr_to_netmask('1.2.3.4/0') == '0.0.0.0'
        assert ipv4.cidr_to_netmask('254.254.254.254/1') == '128.0.0.0'
        assert ipv4.cidr_to_netmask('254.254.254.254/2') == '192.0.0.0'
        assert ipv4.cidr_to_netmask('1.2.3.255/31') == '255.255.255.254'
        assert ipv4.cidr_to_netmask('1.2.3.4/32') == '255.255.255.255'

    def test_non_ipv4(self):
        with self.assertRaises(ValueError):
            ipv4.cidr_to_netmask('1.2.3.555')
        with self.assertRaises(ValueError):
            ipv4.cidr_to_netmask('1.2.3.4/33')
        with self.assertRaises(ValueError):
            ipv4.cidr_to_netmask('1.2.3.4/-1')
        with self.assertRaises(ValueError):
            ipv4.cidr_to_netmask('example.com')


class CidrToIp(unittest.TestCase):
    def test_happy(self):
        assert ipv4.cidr_to_ip('1.2.3.0/24', 44) == '1.2.3.44'
        assert ipv4.cidr_to_ip('1.2.3.45/24', 44) == '1.2.3.44'
        assert ipv4.cidr_to_ip('1.2.3.255/25', 44) == '1.2.3.172'
        assert ipv4.cidr_to_ip('1.2.3.254/31', 1) == '1.2.3.255'
        assert ipv4.cidr_to_ip('1.2.3.255/31', 0) == '1.2.3.254'

    def test_non_ipv4(self):
        with self.assertRaises(ValueError):
            ipv4.cidr_to_ip('1.2.3.555', 0)
        with self.assertRaises(ValueError):
            ipv4.cidr_to_ip('1.2.3.4/33', 0)
        with self.assertRaises(ValueError):
            ipv4.cidr_to_ip('1.2.3.4/-1', 0)
        with self.assertRaises(ValueError):
            ipv4.cidr_to_ip('example.com', 0)


class CidrOrIpv4ToNetwork(unittest.TestCase):
    def test_happy(self):
        assert ipv4.ipv4_or_cidr_to_network('1.2.3.0/24') == '1.2.3.0'
        assert ipv4.ipv4_or_cidr_to_network('1.2.3.45/24') == '1.2.3.0'
        assert ipv4.ipv4_or_cidr_to_network('1.2.3.194/25') == '1.2.3.128'
        assert ipv4.ipv4_or_cidr_to_network('1.2.3.44/25') == '1.2.3.0'
        assert ipv4.ipv4_or_cidr_to_network('1.2.3.254/31') == '1.2.3.254'
        assert ipv4.ipv4_or_cidr_to_network('1.2.3.255/31') == '1.2.3.254'
        assert ipv4.ipv4_or_cidr_to_network('1.2.3.123/32') == '1.2.3.123'

    def test_non_ipv4(self):
        with self.assertRaises(ValueError):
            ipv4.ipv4_or_cidr_to_network('1.2.3.555')
        with self.assertRaises(ValueError):
            ipv4.ipv4_or_cidr_to_network('1.2.3.4/33')
        with self.assertRaises(ValueError):
            ipv4.ipv4_or_cidr_to_network('1.2.3.4/-1')
        with self.assertRaises(ValueError):
            ipv4.ipv4_or_cidr_to_network('example.com')


@patch("socket.gethostbyname")
class HostMatchesAny(unittest.TestCase):
    def test_ipv4_matches_any_ip(self, gethostbyname):
        assert ipv4.host_matches_any('192.168.0.1', ['192.168.0.1']) == True
        assert ipv4.host_matches_any('192.168.0.1', ['192.168.0.0/30']) == True
        assert ipv4.host_matches_any('192.168.1.1', ['192.168.0.0/24']) == False
        gethostbyname.assert_not_called()

    def test_hostname_matches_exact_hostname(self, gethostbyname):
        gethostbyname.return_value = '1.2.3.4'
        assert ipv4.host_matches_any('example.com', ['example.com']) == True
        gethostbyname.assert_called_with('example.com')

    def test_hostname_matches_exact_hostname_but_does_not_exists(self, gethostbyname):
        gethostbyname.side_effect = socket.gaierror
        with self.assertRaises(ValueError):
            ipv4.host_matches_any('example.com', ['example.com'])
        gethostbyname.assert_called_with('example.com')

    def test_hostname_matches_but_smaller(self, gethostbyname):
        gethostbyname.return_value = '1.2.3.4'
        assert ipv4.host_matches_any('a.example.com', ['example.com']) == False
        gethostbyname.assert_not_called()

    def test_hostname_matches_but_bigger(self, gethostbyname):
        gethostbyname.return_value = '1.2.3.4'
        assert ipv4.host_matches_any('example.com', ['a.example.com']) == False
        gethostbyname.assert_not_called()

    def test_hostname_matches_none(self, gethostbyname):
        gethostbyname.return_value = '1.2.3.4'
        assert ipv4.host_matches_any('a.example.com', ['b.example.com', 'c.example.com']) == False
        gethostbyname.assert_not_called()

    def test_hostname_matches_mixed(self, gethostbyname):
        gethostbyname.return_value = '1.2.3.4'
        assert ipv4.host_matches_any('a.example.com', ['b.example.com', 'c.example.com', '1.2.3.4', 'a.example.com']) == True
        gethostbyname.assert_called_with('a.example.com')

    def test_hostname_does_not_match_ips(self, gethostbyname):
        gethostbyname.return_value = '1.2.3.4'
        assert ipv4.host_matches_any('example.com', ['1.2.3.4']) == False
        assert ipv4.host_matches_any('example.com', ['1.2.3.5']) == False
        gethostbyname.assert_not_called()

    def test_invalid_input(self, gethostbyname):
        assert ipv4.host_matches_any('1.2.3.4/24', ['1.2.3.4']) == False
        gethostbyname.assert_not_called()

