# -*- coding: UTF-8 -*-

import os
import glob
from ansibleext.inventory_base import InventoryBase


class PrometheusLabelsRequired(InventoryBase):
    def keysMissing(self, prometheusLabels, requiredKeys):
        return [key for key in requiredKeys if not key in prometheusLabels]

    def isPrometheusDisabled(self, inventory, host):
        prometheusDisabled = self.getVariable(inventory, host, 'prometheus_disabled')
        return prometheusDisabled is not None and prometheusDisabled

    def isNonClusterSpecific(self, inventory, host):
        return self.isHostInAnyGroups(inventory, host, ('nonclusterspecific',))

    def containsMultipleServices(self, inventory, host):
        return self.isHostInAnyGroups(inventory, host, ('kafka', 'tomcat'))

    def isCluster(self, prometheusLabels):
        clusterType = prometheusLabels['cluster_type']
        return clusterType == 'cluster'

    def verifyAlertingHosts(self, prometheusLabels):
        hostLevel = prometheusLabels['host_level']
        team = prometheusLabels['team']
        if hostLevel == 'stable':
            self.assertEqual(team, 'devels',
                             "Team must be devels if host_level is stable, or else alerting will not be configured properly")

    def getRequiredKeys(self, prometheusLabels, inventory, host):
        requiredKeys = self.getBasicRequiredKeys()
        if self.isNonClusterSpecific(inventory, host):
            if self.isCluster(prometheusLabels):
                requiredKeys = requiredKeys + ('cluster_name',)
        else:
            requiredKeys = requiredKeys + ('itrack_cluster',)
        if not self.containsMultipleServices(inventory, host):
            requiredKeys = requiredKeys + ('service_name',)
        return requiredKeys

    def getBasicRequiredKeys(self):
        return ('team', 'host_level', 'cluster_type', 'network_group')

    def verifyMinimumRequiredNodes(self, prometheusLabels, inventory, host):
        clusterType = prometheusLabels['cluster_type']
        if (clusterType == 'cluster'):
            alertConfig = self.getVariable(inventory, host, 'prometheus_unique_alerts')
            self.assertIsNotNone(alertConfig,
                                 "Prometheus cluster_type label is set to 'cluster' on %s but prometheus_unique_alerts is not set" % (
                                     host))
            self.assertIsInstance(alertConfig, dict,
                                  "Prometheus cluster_type label is set to 'cluster' on %s but prometheus_unique_alerts is not a dict" % (
                                      host))
            minRequiredNodes = alertConfig['min_required_nodes']
            self.assertIsNotNone(minRequiredNodes,
                                 "Prometheus cluster_type label is set to 'cluster' on %s but min_required_nodes is not set" % (
                                     host))
            self.assertIsInstance(minRequiredNodes, int, "min_required_nodes is not an int on host {0}".format(host))
            self.assertGreaterEqual(minRequiredNodes, 1, "min_required_nodes has to be >= 1")

    def runTest(self):
        for inventory, host in self.inventories_hosts_onlystable():
            if self.isPrometheusDisabled(inventory, host):
                continue
            prometheusLabels = self.getVariable(inventory, host, 'prometheus_labels')
            self.assertIsInstance(prometheusLabels, dict,
                                  "Prometheus is enabled on %s but prometheus_labels is not a dict" % (host))
            requiredBasicKeys = self.getBasicRequiredKeys()
            missingBasicKeys = self.keysMissing(prometheusLabels, requiredBasicKeys)
            self.assertEqual(missingBasicKeys, [], (
                    'host %s has prometheus_labels, but it lacks the basic labels (more labels could be missing, will be able to show them after basic ones are fixed): %s' % (
                host, missingBasicKeys)))
            requiredKeys = self.getRequiredKeys(prometheusLabels, inventory, host)
            missingKeys = self.keysMissing(prometheusLabels, requiredKeys)
            self.assertEqual(missingKeys, [],
                             ('host %s has prometheus_labels, but it lacks the labels %s' % (host, missingKeys)))
            self.verifyAlertingHosts(prometheusLabels)
            self.verifyMinimumRequiredNodes(prometheusLabels, inventory, host)

class IsPrometheusAlertTemplateLabelFilteringCorrect(InventoryBase):
    def getPrometheusTemplateRules(self):
        rules = glob.glob("playbooks/templates/prometheus/rules/*.rules")
        for rule in rules:
            try:
                data = open(rule, "r")
                yield data
            except OSError as exc:
                self.assertTrue(False, "Exception: %s" % exc)

    # Team label could be more times as by because team is sometimes overriden in alerts and in templates we can't parse as yaml to only check expr variable.
    # Host_level label should be exactly as many times as by
    # Network_group labels can be more times than by, because of jinja templating filtering on network_group variable
    def requiredLabelsPresent(self, byCount, content):
        team = content.count("team")
        hostLevel = content.count("host_level")
        networkGroup = content.count("network_group")
        return (team >= byCount and hostLevel >= byCount and networkGroup >= byCount)

    def runTest(self):
        alertsWithMissingTags = []
        for rule in self.getPrometheusTemplateRules():
            content = rule.read()
            by1 = content.count(" by (")
            by2 = content.count(" by(")
            by3 = content.count(" BY (")
            by4 = content.count(" BY(")
            by = by1 + by2 + by3 + by4
            if by == 0:
                continue
            if not self.requiredLabelsPresent(by, content):
                print(content)
                fileName = os.path.basename(rule.name)
                alertsWithMissingTags.append(fileName)
        numberOfAlertsWithMissingTags = len(alertsWithMissingTags)
        self.assertEqual(numberOfAlertsWithMissingTags, 0,
                         'The following "%s" alert template(s) expression missing either [host_level], [team] or [network_group] tags in: %s' % (
                             numberOfAlertsWithMissingTags, alertsWithMissingTags))
